using System;
using System.IO;
using System.Data;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;
using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.DAL.VAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>1/22/2004</CreationDate>
	///	<Note>
	///		The Food and Drug Administration classifies this software as a medical device.  
	///		As such, it may not be changed in any way. Modifications to this software may result 
	///		in an adulterated medical device under 21CFR820, the use of which is considered to be 
	///		a violation of US Federal Statutes.  Acquiring and implementing this software through 
	///		the Freedom of information Act requires the implementor to assume total responsibility 
	///		for the software, and become a registered manufacturer of a medical device, 
	///		subject to FDA regulations.
	///	</Note>
	/// <summary>
	/// Internally used by the VistALink Access Layer to perform data transformations
	/// during a data exchange via VistALink.
	/// </summary>

	#endregion

	internal class DataTransformUtility
	{
		private DataTransformUtility() {}

		/// <summary>
		/// Makes <see cref="DataSet"/> read-only.
		/// </summary>
		/// <param name="dataSet"><see cref="DataSet"/> to make read-only.</param>
		/// <returns>Reference to supplied <see cref="DataSet"/>.</returns>
		private static DataSet MakeDataSetReadOnly( DataSet dataSet )
		{
			foreach( DataTable _dt in dataSet.Tables ) 
				foreach( DataColumn _column in _dt.Columns ) 
					_column.ReadOnly = true;

			return dataSet;
		}

		/// <summary>
		/// Loads supplied XML string into <see cref="DataSet"/>.
		/// </summary>
		/// <param name="xmlString">XML string to load into <see cref="DataSet"/>.</param>
		/// <returns><see cref="DataSet"/> loaded from XML string.</returns>
		internal static DataSet LoadXmlStringIntoDataSet( string xmlString )
		{
			if( xmlString == null )
				throw( new ArgumentNullException( "xmlString" ) );

			DataSet _ds = new DataSet();
			_ds.ReadXml( new StringReader( xmlString ), XmlReadMode.InferSchema );

			return _ds;
		}

		/// <summary>
		/// Loads supplied XML string into <see cref="DataTable"/>. The XML string must contain data for 
		/// a single <see cref="DataTable"/> only!
		/// </summary>
		/// <param name="xmlString">XML string to load into <see cref="DataTable"/>.</param>
		/// <returns><see cref="DataTable"/> loaded from XML string.</returns>
		internal static DataTable LoadXmlStringIntoSingleDataTable( string xmlString )
		{
			if( xmlString == null )
				throw( new ArgumentNullException( "xmlString" ) );

			DataSet _ds = LoadXmlStringIntoDataSet( xmlString );

			if( _ds.Tables.Count != 1 )
				throw( new VistALinkException( StrRes.SysErrMsg.VAL.DTUInvalidNumberOfDataTablesFoundAfterXmlConversion( _ds.Tables.Count ).ResString ) );

			DataTable _dt = _ds.Tables[0];
			_ds.Tables.RemoveAt( 0 ); // removing internal reference to DataSet to make sure it won't be hanging around
			
			return _dt;
		}

		/// <summary>
		/// Loads supplied XML string into <see cref="DataTable"/>. String is converted to <see cref="DataSet"/> 
		/// first and then <see cref="DataTable"/> with a given name is returned.
		/// </summary>
		/// <param name="xmlString">XML string to load into <see cref="DataTable"/>.</param>
		/// <param name="dataTableName">Name of <see cref="DataTable"/> that must be created from supplied XML string.</param>
		/// <returns><see cref="DataTable"/> loaded from XML string.</returns>
		internal static DataTable LoadXmlStringIntoNamedDataTable( string xmlString, string dataTableName )
		{
			if( xmlString == null )
				throw( new ArgumentNullException( "xmlString" ) );

			if( dataTableName == null )
				throw( new ArgumentNullException( "dataTableName" ) );

			DataSet _ds = LoadXmlStringIntoDataSet( xmlString );
			DataTable _dt = _ds.Tables[ dataTableName ];
			
			if( _dt == null )
				throw( new VistALinkException( StrRes.SysErrMsg.VAL.DTUNamedDataTableNotFoundAfterXmlConversion( dataTableName ).ResString ) );

			_ds.Tables.Remove( _dt ); // removing internal reference to DataSet to make sure it won't be hanging around

			return _dt;
		}

		/// <summary>
		/// Loads supplied XML string into <see cref="DataSet"/> and makes the latter read-only.
		/// </summary>
		/// <param name="xmlString">XML string to load into read-only <see cref="DataSet"/>.</param>
		/// <returns>Read-only <see cref="DataSet"/> loaded from XML string.</returns>
		internal static DataSet LoadXmlStringIntoReadOnlyDataSet( string xmlString )
		{
			if( xmlString == null )
				throw( new ArgumentNullException( "xmlString" ) );

			return MakeDataSetReadOnly( LoadXmlStringIntoDataSet( xmlString ) );
		}
	}
}
